const CACHE_VERSION = 2
const CACHE_NAME = `editorV2-texture-cache-v${CACHE_VERSION}`
const MAX_ENTRIES = 300
const ALLOW_OPAQUE = true
const ALLOW_ORIGINS = [self.location.origin]
const EXTENSIONS = ['.png','.jpg','.jpeg','.webp','.gif','.ktx','.ktx2','.dds','.env','.basis','.hdr']

const shouldCache = (req) => {
  try {
    if (req.method !== 'GET') return false
    const url = new URL(req.url)
    const originOk = ALLOW_ORIGINS.includes(url.origin) || ALLOW_OPAQUE
    if (!originOk) return false
    const dest = req.destination
    if (dest === 'image') return true
    const p = url.pathname.toLowerCase()
    return EXTENSIONS.some((e) => p.endsWith(e))
  } catch { return false }
}

const limitCacheSize = async (cache) => {
  try {
    const keys = await cache.keys()
    const extra = keys.length - MAX_ENTRIES
    if (extra > 0) {
      for (let i = 0; i < extra; i++) { await cache.delete(keys[i]) }
    }
  } catch {}
}

self.addEventListener('install', (event) => {
  self.skipWaiting()
  event.waitUntil(caches.open(CACHE_NAME))
})

self.addEventListener('activate', (event) => {
  event.waitUntil((async () => {
    const keys = await caches.keys()
    await Promise.all(keys.map((k) => (k !== CACHE_NAME ? caches.delete(k) : Promise.resolve())))
    await self.clients.claim()
  })())
})

self.addEventListener('fetch', (event) => {
  const req = event.request
  if (!shouldCache(req)) return
  event.respondWith((async () => {
    const cache = await caches.open(CACHE_NAME)
    const cached = await cache.match(req)
    if (cached) {
      event.waitUntil((async () => {
        try {
          const res = await fetch(req)
          if (res && (res.status === 200 || res.type === 'opaque')) {
            await cache.put(req, res.clone())
            await limitCacheSize(cache)
          }
        } catch {}
      })())
      return cached
    }
    try {
      const res = await fetch(req)
      if (res && (res.status === 200 || res.type === 'opaque')) {
        await cache.put(req, res.clone())
        await limitCacheSize(cache)
      }
      return res
    } catch (e) {
      return cached || Promise.reject(e)
    }
  })())
})

self.addEventListener('message', (event) => {
  const data = event.data || {}
  const type = data.type
  if (type === 'CACHE_CLEAR') {
    event.waitUntil((async () => { await caches.delete(CACHE_NAME) })())
  } else if (type === 'CACHE_DELETE_URL') {
    const u = String(data.url || '')
    if (!u) return
    event.waitUntil((async () => {
      const cache = await caches.open(CACHE_NAME)
      const req = new Request(u, { method: 'GET' })
      await cache.delete(req)
    })())
  }
})
